<?php
/* spatz-base.inc.php
 * Author: Dennis 'Gyroplast' Herbrich <herbrich@veloxis.de>
 *
 * This software is released under the GNU General Public License (GPL).
 *
 * CHANGELOG:
 * 02. December 2007 v1.0.0: Initial release
 *
 * Include this file where you need the announcements to appear, for example 
 * with a statement like this:
 *
 *  <?php
 *    include_once("spatz-base.inc.php");
 *  ?>
 *
 * This will allow you to use the function named 
 * spatz_generate_announcements() whereever you need the actual
 * HTML announcements to appear. This function returns TRUE if everything went 
 * fine, and FALSE if a fatal, non-recoverable error occured.
 * Currently the only way to cause such a fatal error is an invalid path to the 
 * announce source file, everything else will only log warnings.
 *
 * Please note that a fatal error causes no output, so if you want a visible 
 * error message in your HTML page, wrap some if-clause around the call to this 
 * function to correctly handle the return value.
 *
 * An inclusion example in your main page may look like this:
 *
 * <?php 
 *   if (FALSE === spatz_generate_announcements()) {
 *     echo "\n<p>\n";
 *     echo '<span style="color: red">&gt;&gt;&gt;</span>&nbsp;';
 *     echo 'Spatz encountered an unrecoverable error. Sorry.';
 *     echo "\n</p>\n";
 *   }
 * ?>
 *
 * BUGS:
 * Spatz will react funny if you use single underscores or asterisks in your 
 * announce source file; It does not check for well-formedness. 
 * Encode these with &#42; (asterisk) or &#95; (underscore), respectively, or 
 * prefix them with a single backslash to escape their special meaning, as 
 * in \_ or \*. You can also escape more than one: \*\*\* is perfectly valid,
 * and will display *** instead of starting a primary heading.
 *
 * APPEARANCE:
 * To modify the announcement appearance on your page, use the 
 * div#spatz_announcement identifier in your CSS rules as follows:
 *
 * div#spatz_announce        : DIV container for the announcements
 * div#spatz_announce h1     : primary headings
 * div#spatz_announce h2     : secondary headings
 * div#spatz_announce em     : _emphasized text_
 * div#spatz_announce b      : *bold faced text*
 * div#spatz_announce hr     : embedded horizontal rulers
 * div#spatz_announce a      : normal hyperlinks
 * div#spatz_announce a.mail : mailto hyperlinks
 *
 * MARKUP:
 * To learn how to markup your text files properly, please refer to the example 
 * announcement source file "announce.txt" included in the distribution 
 * package, or available online at http://misc.veloxis.de/spatz/announce.txt
 *
 * Have fun!
 */

/* PROGRAM SETTINGS */

// full path to your announcement text file. Implicitly sets the path to a file
// saving a hash value of the last successfully converted source file, which is 
// "SOURCE_FILE.hash". This hash file does not have to exist, but should be 
// writable by the user running this PHP process.
//
// NOTE: The SOURCE_FILE *must* exist and be readable by the PHP process user, 
// otherwise the script will fail! An absolute path is preferred.
define("SOURCE_FILE","./announce.txt");

// full path to file to be used as a cache. Does not have to exist, but should be 
// writable by the user running this PHP process.
define("TARGET_FILE","./announce-converted.txt"); 

// set to \n for linux line delimiters, \r\n for windows, and \r for macintosh. 
// When in doubt, try out all of them until it works. :)
define("ENDLINE", "\n");

define("DEBUG", FALSE); // generate verbose debugging messages?

/************* Nothing to change by users below this point - Dragons and foxes live here! **************/

function debug_msg($msg) {
  if (TRUE == DEBUG) {
    trigger_error($msg, E_USER_NOTICE);
  }
  return TRUE;
}

function spatz_generate_announcements() {

  debug_msg("Spatz starting");
      
  if (FALSE === is_readable(SOURCE_FILE)) {
    trigger_error("Announce source file \"" . SOURCE_FILE . "\" can not be read",E_USER_WARNING);
    return(FALSE); // if annouce file can not be read, bail with error
  }
  debug_msg("Announce source file \"" . SOURCE_FILE . "\" is readable");

  $hash = md5_file(SOURCE_FILE); // store hash of source file for later
  debug_msg("Hash of source file: \"$hash\"");

  if (TRUE === @is_readable(SOURCE_FILE . ".hash")) { // check hash file
    $stored_hash =  file_get_contents(SOURCE_FILE . ".hash");
    debug_msg("Stored hash of lastly converted source file: \"$stored_hash\"");

    if ($stored_hash && $hash && ($stored_hash == $hash)) {
      debug_msg("Hashes considered identical");
      // annouce file has not been changed -> pass through existing converted file
      if (FALSE === @is_readable(TARGET_FILE)) {
        trigger_error("Announce target file \"" . TARGET_FILE . "\" can not be read", E_USER_WARNING);
      } else {
        debug_msg("Passing through contents of target file \"".TARGET_FILE."\"");
	@readfile(TARGET_FILE);
	debug_msg("Spatz terminating early");
	return(TRUE);
      }
    }
  }

  // hash is different -> announce file has been changed. Convert text to html.
  $content = file_get_contents(SOURCE_FILE); // file is readable, tested already
  if (FALSE === $content) {
    trigger_error("Content of announce source file \"" . SOURCE_FILE . "\" can not be read",E_USER_ERROR);
    exit(2);
  }
  debug_msg("Content of announce source file \"".SOURCE_FILE."\" successfully read");

  $mapping = array();
  $mapping['/'.ENDLINE.'{2}/'] = ENDLINE.'<br /><br />'.ENDLINE;
  $mapping['/\*{3}(.+)\*{3}/sU'] = ENDLINE.'<h1>\\1</h1>'.ENDLINE;
  $mapping['/\*{2}(.+)\*{2}/sU'] = ENDLINE.'<h2>\\1</h2>'.ENDLINE;
  $mapping['/\\\\\_/U'] = '&#95;';
  $mapping['/\\\\\*/U'] = '&#42;';
  $mapping['/\*{1}(.+)\*{1}/sU'] = '<b>\\1</b>';
  $mapping['/\_{1}(.+)\_{1}/sU'] = '<em>\\1</em>';
  $mapping['/\s*\-{5}\s*/U'] = '<hr />';
  $mapping['/\[{1}(\S+\:\/\/\S+)\]{1}/U'] = '<a href="\\1">\\1</a>';
  $mapping['/\[{1}(\S+\:\/\/\S+)\s+(.*)\]{1}/sU'] = '<a href="\\1">\\2</a>';
  $mapping['/\[{1}(mailto\:(\S+))\]{1}/iU'] = '<a class="mail" href="\\1">\\2</a>';
  $mapping['/\[{1}(mailto\:\S+)\s+(.*)\]{1}/isU'] = '<a class="mail" href="\\1">\\2</a>';

  $converted_content = preg_replace(array_keys($mapping), array_values($mapping), $content);

  $announce_html  = '<div id="spatz_announce">' . "\n";
  $announce_html .= $converted_content;
  $announce_html .= '</div>' . "\n";

  echo $announce_html;

  // save new converted text to file
  $write_handle = @fopen(TARGET_FILE, "wb");
  if (FALSE === $write_handle) {
    trigger_error("Could not create or open target file \"".TARGET_FILE."\"", E_USER_WARNING);
  } elseif (FALSE === @fwrite($write_handle, $announce_html)) {
    trigger_error("Could not write to target file \"".TARGET_FILE."\"", E_USER_WARNING);
  } else {
    // converted content has successfully been written to file -> close it and discard handle
    debug_msg("Generated HTML successfully written to target file \"".TARGET_FILE."\"");
    fclose($write_handle);
    unset($write_handle);
  }

  // save new hash to file
  $write_handle = @fopen(SOURCE_FILE . ".hash", "wb");
  if (FALSE === $write_handle) {
    trigger_error("Could not create or open hash file \"" . SOURCE_FILE . ".hash\"", E_USER_WARNING);
  } elseif (FALSE === @fwrite($write_handle, $hash)) {
    trigger_error("Could not write to hash file \"" . SOURCE_FILE . ".hash\"", E_USER_WARNING);
  } else {
    // hash has successfully been written to file -> close it and discard handle
    debug_msg("New hash successfully written to hash file \"".SOURCE_FILE.".hash\"");
    fclose($write_handle);
    unset($write_handle);
  }
  debug_msg("Spatz done");
  return(TRUE);
}
?>
